#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
 This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
  
May 2018 
    
cpt.py

A class for representing conditional probability tables

@author: Stan (stalinmunoz@yahoo.com)
"""
from itertools import product as cartesian
from functools import reduce
from literal import Literal
# Conditional probability table
class CPT:
    # initializes the condional probability table for the given variable
    # table, a list of pairs (v,p)
    # where v is a string with a comma separated list of literals
    # and p the corresponding probability
    # for example:
    # [("~A,~B",0.5),("~A,B",0.8),("A,~B",0.3),("A,B",0.9)]
    def __init__(self,variable,table):
        self.variable = variable
        #parse variables from first row of table
        self.parents = CPT.parseVariables(table[0][0])
        self.fillDictionary(table)
        
    # parses the variables from the first row of the table
    @staticmethod
    def parseVariables(s):
        return set([l.variable for l in CPT.stringToLiteralList(s)])

    # obtains all variations of the given variables
    @staticmethod
    def computeKeys(variables):
        entries = []
        if len(variables) == 1:
            (v,)= variables
            keys = [CPT.formatLiteral(Literal(variable=v,value=False)),\
                    CPT.formatLiteral(Literal(variable=v,value=True))] 
        else:            
            for e in [CPT.flatten(i,[]) \
            for i in list(reduce(lambda x,y:cartesian(x,y), \
            [cartesian([v],[False,True]) for v in sorted(variables)]))]:
                entries.append([Literal(variable=a[0],value=a[1]) for a in e])
            keys = [reduce(lambda x,y: \
            CPT.formatLiteral(x)+','+CPT.formatLiteral(y),e) for e in entries]
        return keys
    
    @staticmethod
    def formatLiteral(l):
        return str(l).rjust(len(str(l))+1) \
        if l.value else str(l).rjust(len(str(l)))
        
    # fills the conditional probability table
    def fillDictionary(self,table):
        self.table = {self.hashString(row[0]):row[1] for row in table}
    
    # converts the string to a list of literals
    @staticmethod
    def stringToLiteralList(s):
        l=[] 
        # there are parents
        if s:
            l = [Literal(l) for l in s.split(",")]
        return l
    
    # hashes the provided string (list of literals)
    def hashString(self,s):
        return hash(frozenset(self.stringToLiteralList(s)))
    
    # allows indexing of the table
    def __getitem__(self,s):
        return self.table[self.hashString(s)]
    
    # allows changing probabilities in the table
    def __setitem__(self,s,p):
            self.table[self.hashString(s)] = p
            
    def __repr__(self):
        return self.__str__()
    
    # pretty printing of the conditional probability table
    def __str__(self):
        s = "CPT for variable "+self.variable+"\n"
        if self.parents:
            for key in CPT.computeKeys(self.parents):
                s += "P("+self.variable+"|"+key+") = "\
                +str("%.4f" % self[key]) + "\n"
        else:
            s += "P("+self.variable+ ") = " + str("%.4f" % self['']) + "\n"
        return s
        
    # flattens a nested tuple
    @staticmethod
    def flatten(x,l=[]):
        if not isinstance(x[0],tuple):
            l.extend([x])
        else:
            for i in x:
                CPT.flatten(i,l)
        return l