import pandas as pd
from dash import Dash, Input, Output, dcc, html, callback
import plotly.graph_objects as go
import plotly.express as px
import plotly.figure_factory as ff
import requests
import numpy as np
from plotly.subplots import make_subplots
from sklearn.cluster import KMeans
from scipy import stats
from ast import literal_eval

import flask

server = flask.Flask(__name__)



def datavis(datos):
    medidas = ["imc", "age", "glucose", "hba1c", "ct", "hdl"]    
    fig = make_subplots(rows=2, cols=3, vertical_spacing=0.25)
    lcol = 1
    lrow = 1
    seq = 1
    for medida in medidas:
        fig.add_trace(
            go.Scatter(x=datos["id"], y=datos[medida], name=medida,mode="markers"),
            row=lrow, col=lcol
        )
        fig['layout']['xaxis' + str(seq)]['title']='Patient ID'
        fig['layout']['yaxis' + str(seq)]['title']=medida     
        seq = seq + 1
        lrow = lrow + (lcol // 3)
        lcol = lcol + 1 if lcol < 3 else 1
    fig.update_layout(showlegend = False)
    return fig

def popdist(datos):
    medidas = ["imc", "age", "glucose", "hba1c", "ct", "hdl"]
    # remoemos outliers
    fig = make_subplots(rows=2, cols=3, vertical_spacing = 0.25 )
    lcol = 1
    lrow = 1
    seq = 1
    for medida in medidas:
        datosfil = datos[np.abs(stats.zscore(datos[medida], nan_policy='omit')) < 3]
        fig.add_trace(
            go.Histogram(x=datosfil[medida], name=medida),
            row=lrow, col=lcol
        )
        fig['layout']['yaxis' + str(seq)]['title']=medida     
        seq = seq + 1
        
        lrow = lrow + (lcol // 3)
        lcol = lcol + 1 if lcol < 3 else 1

    fig.update_layout(
        bargap=0.05, # gap between bars of adjacent location coordinates
        bargroupgap=0.05, # gap between bars of the same location coordinates
        showlegend = False
    )
    return fig

def popcorr(datos):
    medidas = [ "glucose", "hba1c", "ct", "hdl", "insuline"]
    # removemos outliers
    fig = make_subplots(rows=2, cols=3, shared_yaxes=False, vertical_spacing=0.25)
    lcol = 1
    lrow = 1
    seq = 1
    for medida in medidas:
        datosfil = datos[np.abs(stats.zscore(datos[medida], nan_policy='omit')) < 3]
        fig.add_trace(
            go.Scatter(x=datosfil["imc"], y=datosfil[medida], name=medida, mode="markers"),
            row=lrow, col=lcol
        )
        fig['layout']['xaxis' + str(seq)]['title']='IMC'
        fig['layout']['yaxis' + str(seq)]['title']=medida        
        lrow = lrow + (lcol // 3)
        lcol = lcol + 1 if lcol < 3 else 1
        seq = seq + 1

    fig.update_layout(
        bargap=0.05, # gap between bars of adjacent location coordinates
        bargroupgap=0.05, # gap between bars of the same location coordinates
        showlegend = False
    )

    return fig



def imc_cat(datos):
    medidas = [ "glucose", "hba1c", "ct", "hdl", "insuline"]
    grupos = ["LW","NW","OW","O-CI","O-CII"]
    datos["imc_cat"] = pd.cut(datos["imc"], bins=[0,18,25,30,35,40], labels=grupos)
    fig = make_subplots(rows=2, cols=3, shared_yaxes=False, vertical_spacing=0.25)
    lcol = 1
    lrow = 1
    seq = 1
    for medida in medidas:
        datosfil = datos.groupby("imc_cat")[medida].mean()
        fig.add_trace(
            go.Scatter(x=grupos, y=datosfil, name=medida),
            row=lrow, col=lcol
        )
        fig['layout']['yaxis' + str(seq)]['title']=medida                
        lrow = lrow + (lcol // 3)
        lcol = lcol + 1 if lcol < 3 else 1
        seq = seq + 1

    fig.update_layout(
        showlegend = False
    )

    return fig

def imc_cat_bigote(datos):
    medidas = [ "glucose", "hba1c", "ct", "hdl", "insuline"]
    grupos = ["LW","NW","OW","O-CI","O-CII"]
    datos["imc_cat"] = pd.cut(datos["imc"], bins=[0,18,25,30,35,40], labels=grupos)
#    figura = px.box(datos, x="imc_cat", y="glucose")
    fig = make_subplots(rows=2, cols=3, vertical_spacing=0.25)
    lcol = 1
    lrow = 1
    seq = 1
    for medida in medidas:
        fig.add_trace(
            go.Box(x=datos["imc_cat"], y=datos[medida]),
            row=lrow, col=lcol
        )
        fig['layout']['yaxis' + str(seq)]['title']=medida                
        lrow = lrow + (lcol // 3)
        lcol = lcol + 1 if lcol < 3 else 1
        seq = seq + 1

    fig.update_xaxes(categoryorder='array', categoryarray=grupos)
    fig.update_layout(
        showlegend = False
    )
    return fig




def correlaciones_v2(datos, controlado):
# filtramos solo a los que tienen datos del controlado
    datos_fil2 = datos.loc[datos[controlado].notnull()]


    metricas = ["imc", "age", "glucose", "hba1c"]
    figura = make_subplots(rows=2, cols=2, shared_xaxes=True, vertical_spacing=0.1, subplot_titles=metricas)
    lrow = 1
    lcol = 1
    for metrica in metricas:
        datos_sorted = datos_fil2.sort_values(by=metrica, ascending=True)
        datos_sorted.reset_index(inplace=True)
        datosfil = datos_sorted[np.abs(stats.zscore(datos_sorted[controlado], nan_policy='omit')) < 3]     

        figura.add_trace(
            go.Scatter(x=datosfil.index, y=datosfil[controlado], name=metrica, mode='markers'),
            row=lrow, col=lcol
        )
        figura.add_trace(
            go.Scatter(x=datosfil.index, y=datosfil[controlado].rolling(5).mean(), name="PM(5)"),
            row=lrow, col=lcol
        )
        corr = datos_sorted[[metrica]].corrwith(datos_sorted[controlado]).values[0]
        figura.add_annotation(row=lrow, col=lcol, text="R = " + "{:.2f}".format(corr))
        lcol = lcol + 1 if lcol < 2 else 1
        lrow = lrow + (lcol % 2)
    figura.update_layout(showlegend=False)

    return figura
    
def controlado_ppgr(datos):

    controlados = ["Glucosa", "Gelatina", "Pan"]
    datos_fil = datos[controlados]
    datos_fil["id"] = datos["id"]
    datos_fil = datos_fil[~datos_fil.isnull().any(axis=1)]    
    fig = make_subplots(rows=1, cols=3)
    lcol = 1
    seq = 1
    for controlado in controlados:
        fig.add_trace(
            go.Scatter(x=datos_fil["id"],y=datos_fil[controlado], name=controlado, mode="markers"),
            row = 1, col = lcol
        )
        fig['layout']['yaxis' + str(seq)]['title']=controlado
        fig['layout']['xaxis' + str(seq)]['title']="Patient ID"                        
        lcol = lcol + 1
        seq = seq + 1
        
    fig.update_layout(
        showlegend = False,
    )
    return fig

def controlado_ppgr2(datos):
    controlados = ["Glucosa", "Gelatina", "Pan"]
    datos_fil = datos[controlados]
    datos_fil = datos_fil[~datos_fil.isnull().any(axis=1)]

    figura = ff.create_distplot([ datos_fil.loc[datos_fil[controlado]!=0, controlado] for controlado in controlados], controlados, show_hist=False, show_rug=False)
    figura.update(layout_title_text='Kernel Density Estimation')
    figura["layout"]["xaxis"]["title"] = "iAUC"
    return figura

def triareas(pacientes):
    figura = make_subplots(rows=1,cols=1)
    for paciente in pacientes:
        
        url_datos = url + "patients/" + str(paciente) + "/visit-data/1"
        datos = requests.get(url_datos).json()["tolerance_curve_measure"][0]
        valores = list(datos.values())
        alimento = valores.pop(0)
        iauc = valores.pop(-1)
        figura.add_trace(
            go.Scatter(x=[0,15,30,45,60,90,120], y= valores, name=str(paciente)+":"+str(iauc)),
            row = 1, col=1
        )
    return figura







def heatmap(datos):

    datosfil = datos.replace(0,np.nan)
    datosfil.dropna(inplace=True, subset=["Gelatina", "Pan", "Glucosa"])
    ranks = datosfil[["Pan", "Gelatina", "Glucosa"]].rank()
    model = KMeans(n_clusters=3)
    model.fit(ranks)
    all_predictions = model.predict(ranks)
    datosfil["grupo"] = all_predictions

    datosfil.sort_values(by="grupo", inplace=True)
    fig = go.Figure(
        data = go.Heatmap(
            z=[datosfil["Gelatina"], datosfil["Pan"], datosfil["Glucosa"]]
        )
    )
    return fig

    
        
        
# algunos parámetros globales
url = 'https://nutricion.c3.unam.mx/nd/'


def get_visit_data(visitnumber):
    url_visitas = url + "/visits/" + str(visitnumber) + "/patients-data/"
#    url_visitas = "https://nutricion.c3.unam.mx/nd/visits/1/patients-data/"
    try:
        yaisons = requests.get(url_visitas).json()["patient_data"]
    except Exception as e:
        print("error al traer datos desde ", url_visitas)
        print(e)
        exit()
        
    df = pd.json_normalize(yaisons)
    df.drop(["tolerance_curve_measure","name","patient_visit_id", "visit_date"], inplace=True, axis=1)
    lista = [ { cf["controlled_food"]: cf["increment_auc"] for cf in yaison["tolerance_curve_measure"] } for yaison in yaisons ] 
    df.rename(columns= lambda x: x.split(".")[1] if "." in x else x, inplace=True)
    #df.drop(["patient_visit_id"], inplace=True, axis=0) #esta etiqueta aparece dos veces: dentro del primer nivel de json y dentro del json de sample
    return pd.concat([df, pd.DataFrame(lista)], axis=1)

# fuentes de datos

# consulta online a la API para las visitas, que proporcionan la info de
# visitas y microbiota
visitas_datos = [ get_visit_data(visita) for visita in range(1, 5)]
comidas = pd.read_csv("todos-foods.csv", usecols=["hc_total", "kcal_total", "fiber_total",\
                                                  "protein_total", "lipids_total","fecha_hora", "patient", "visita",\
                                                  "glucosa_basal_area", "glucosa_estimulo_area",\
                                                  "glucosa_area_efectiva","glucosa_area_relativa","foods","dominante", "dominante_category"])

comidas["foods"] = comidas["foods"].apply(literal_eval)

comidas = comidas[comidas["visita"] == "e1"]
comidas = comidas[comidas["hc_total"]>0]
comidas.rename(columns={"glucosa_basal_area":"pre-AUC", "glucosa_estimulo_area":"post-AUC", "glucosa_area_efectiva":"iAUC"}, inplace=True)
comidas["iAUC"] = comidas["post-AUC"] - comidas["pre-AUC"] * 2
print(comidas.columns)
# la info de glucómetro y comidas es más compleja. por el momento, se estab armando
# la info offline.

data_o = (
    pd.read_csv("inventario.csv")
)

data_o["paciente"] = data_o["paciente"].astype("str") + "_p"
data_o["Secuenciaciones"] = 0


Total_pacientes = len(data_o.index)
data = data_o.loc[data_o["incluido"]]
Total_incluidos = len(data)
# gráficas de distribución por indicadores.

visitas_incluidos = visitas_datos[0].loc[visitas_datos[0]["include_in_analysis"]]


    

# grafica estática de visitas/secuenciaciones
scat = px.scatter(data, x="NDias_glucosa", y="Comidas_Aisladas",size="AUCs_glucosa")
vis_sec = {
        "data": [
            {
                #"x": temp["paciente"],
                "x": data["Nvisitas"].astype("str") + "_visitas",
                "type": "histogram",
                "name": "Visitas",
                "histnorm": "percent",                
                #"orientation": "v",
            },
            { "x": data["Secuenciaciones"], "type": "histogram", "name": "Secuenciaciones", "histnorm": "percent"},
        ],
        "layout": {
            #"title": {"text": title + " " + etapalab + str(etapaDet)},
            "xaxis": { "title": {"text": "Visitas por paciente"}},
            "yaxis": { "title": {"text": "% de pacientes"}},            
        }
    }


external_stylesheets = [
    {
        "href": (
            "https://fonts.googleapis.com/css2?"
            "family=Lato:wght@400;700&display=swap"
        ),
        "rel": "stylesheet",
    },
]

app = Dash(__name__, external_stylesheets=external_stylesheets, server=server, url_base_pathname='/nutri/estadisticas/')
app.title = "NutrIndMex. Análisis de datos."

app.layout = html.Div(children=[
    #encabezado
    html.Div(children=[
        html.P(children="🥑", className="header-emoji"),
        html.H1(children="NutrIndMex", className="header-title"),
        html.H2(children="Análisis de Datos", className="header-title"),

    ],
    className="header",),
    # descripción por diferentes indices
    html.Div(children=[
        html.Div(children= [ html.H1("Características de la población", className= "header-title2"),], ),
        dcc.Graph(figure=datavis(visitas_incluidos))
    ]),

    # descripción por diferentes indices
    html.Div(children=[
        html.Div(children= [ html.H1("Distribuciones de la población", className= "header-title2"),], ),
        dcc.Graph(figure=popdist(visitas_incluidos)),

    ]),
    # correlaciones entre indices de la poblacion
    html.Div(children=[
        html.Div(children= [ html.H1("QS vs IMC", className= "header-title2"),], ),
        dcc.Graph(figure=popcorr(visitas_incluidos)),
    ]),
    html.Div(children=[
        html.Div(children= [ html.H1("QS vs grupo IMC (promedios)", className= "header-title2"),], ),
        dcc.Graph(figure=imc_cat(visitas_incluidos)),
    ]),
    html.Div(children=[
        html.Div(children= [ html.H1("QS vs grupo IMC (boxes)", className= "header-title2"),], ),
        dcc.Graph(figure=imc_cat_bigote(visitas_incluidos))        
    ]),

    # ppgr a controlados
    html.Div(children= [
        html.Div(children= [html.H1("iAUC de alimentos controlados ", className = "header-title2")]),
        dcc.Graph(figure=controlado_ppgr(visitas_incluidos)),
        dcc.Graph(figure=controlado_ppgr2(visitas_incluidos))
    ]),

    # correlaciones
    html.Div(children= [
        html.Div(children= [html.H1("Relación de iAUC/Controlados con diversos índices", className = "header-title2")]),
        html.Div(children=[
            dcc.Dropdown(
                id="selControlado",
                options=[
                    {"label":"Pan", "value":"Pan"},
                    {"label":"Gelatina", "value":"Gelatina"},
                    {"label":"Glucosa", "value":"Glucosa"},
                ],
                value="Gelatina",
                clearable=False,
                searchable=False,
                className="dropdown",
            ),        
        ], className="menu-title2"),
        dcc.Graph(id="figCorrelaciones")
    ]),
    # correlaciones transpuesta
    # html.Div(children= [
    #     html.Div(children= [html.H1("Relación de iAUC/Controlados con diversos índices (traspuesta)", className = "header-title2")]),
    #     dcc.Graph(figure=correlaciones_v2(visitas_incluidos, "Glucosa"))
    # ]),

    # areas de tres pacientes
    html.Div(children= [
        html.Div(children= [html.H1("Ejemplos de iAUC de Glucosa", className = "header-title2")]),
        dcc.Graph(figure=triareas([85,148,177]))
    ]),
    # # heatmap
    # html.Div(children= [
    #     html.Div(children= [html.H1("Mapa de calor de iAUC/alimentos controlados", className = "header-title2")]),
    #     dcc.Graph(figure=heatmap(visitas_incluidos))
    # ,]),

]) # cierre del layout





@callback(
    Output("figCorrelaciones", "figure"),
    Input("selControlado", "value"),
)  

def update_correlacion(controlado):

# filtramos solo a los que tienen datos del controlado
    datos = visitas_incluidos
    datos_fil = datos.loc[datos[controlado].notnull()]
    datos_sorted = datos_fil.sort_values(by=controlado, ascending=True)
    datos_sorted.reset_index(inplace=True)
    metricas = ["imc", "age", "glucose", "hba1c"]
    figura = make_subplots(rows=1, cols=4, vertical_spacing=0.25)
    lrow = 1
    lcol = 1
    seq = 1
    for metrica in metricas:
        datosfil = datos_sorted[np.abs(stats.zscore(datos_sorted[metrica], nan_policy='omit')) < 3]     

        figura.add_trace(
            go.Scatter(x=datos_sorted.index, y=datos_sorted[metrica], name=metrica, mode='markers'),
            row=1, col=lcol
        )
        figura.add_trace(
            go.Scatter(x=datos_sorted.index, y=datos_sorted[metrica].rolling(5).mean(), name="PM(5)"),
            row=1, col=lcol
        )
        corr = datos_sorted[[metrica]].corrwith(datos_sorted[controlado]).values[0]
        figura.add_annotation(row=1, col=lcol, text="R = " + "{:.2f}".format(corr))
        figura["layout"]['xaxis'  + str(seq)]['title']="Ranqueo de pacientes por iAUC: " + controlado
        figura["layout"]['yaxis'  + str(seq)]['title']= metrica
        lcol = lcol + 1
        seq = seq + 1
    figura["layout"]["yaxis1"]["range"] = [15,45]
    figura["layout"]["yaxis2"]["range"] = [17,80]
    figura["layout"]["yaxis3"]["range"] = [60,180]
    figura["layout"]["yaxis4"]["range"] = [5,7]    
    
    figura.update_layout(showlegend=False)
    #figura.update_layout(height=600, width=800)    
    return  figura

if __name__ == "__main__":
    app.run_server(debug=True, port=8052, host='0.0.0.0')

