import pandas as pd
from dash import Dash, Input, Output, dcc, html
import plotly.graph_objects as go
import plotly.express as px
from plotly.subplots import make_subplots
import requests
import numpy as np
import flask
import test_ch_2_b

def get_visit_data(visitnumber):
    url_visitas = url + "/visits/" + str(visitnumber) + "/patients-data"
    yaisons = requests.get(url_visitas).json()["patient_data"]
    df = pd.json_normalize(yaisons)
    df.drop(["tolerance_curve_measure","name","patient_visit_id", "visit_date"], inplace=True, axis=1)
    df.drop(["sex", "age", "height",], inplace=True, axis=1) # estas ya vienen en 'pacientes'.
    lista = [ { cf["controlled_food"]: cf["increment_auc"] for cf in yaison["tolerance_curve_measure"] } for yaison in yaisons ] 
    df.rename(columns= lambda x: x.split(".")[1] if "." in x else x, inplace=True)
    df["muestra"] = df["patient_visit_id"] 
    df = pd.concat([df, pd.DataFrame(lista)], axis=1)
    return df.loc[df["include_in_analysis"]]
    return pd.concat([df, pd.DataFrame(lista)], axis=1)

def get_secuenciaciones_data(visitnumber=1):
    secuenciaciones = pd.read_json(f'{url}/microbiota/7/organisms-quantification/{visitnumber}')
    numero_de = secuenciaciones["patient_id"].unique().size
    return numero_de


server = flask.Flask(__name__)

# algunos parámetros globales

url = 'https://nutricion.c3.unam.mx/nd/'
Visitas_e = 4
Etapas_e = 3
Comidas_E1_e = 42
Comidas_E2_e = 9
Comidas_E3_e = 9
#Comidas_totales_e = Comidas_E1_e + Comidas_E2_e + Comidas_E3_e
Comidas_totales_e = 42
Glucometro_e = 14

esperados = {
    "antrop-mets": 3,
    "quimica-S": 13,
    "comp-corp":  6,
    "physiological": 10,
    "tolerancia": 3
}


# fuentes de datos
# consulta online de la población y selección de los que si participaron.
reclutados = pd.read_json(url+"patients/")
reclutados.loc[~reclutados["include_in_analysis"], "finished_periods"] = 0
participantes = reclutados.loc[reclutados["include_in_analysis"]]
excluidos = reclutados.loc[~reclutados["include_in_analysis"],"id"]


# la info de glucómetro y comidas viene de clickhouse_connect.
comidas_df = test_ch_2_b.foods()
excluye = comidas_df.loc[comidas_df["patient"].isin(excluidos)].index
comidas_df.drop(labels=excluye, axis=0, inplace=True)
# creamos un dataframe que tiene como indice (patient, etapa) con el número respectivo de comidas.
resumen_comidas = comidas_df.groupby(["patient","etapas"]).count()["Source"]

glucometro_df = test_ch_2_b.glucometros()
excluye = glucometro_df.loc[glucometro_df["patient_id"].isin(excluidos)].index
glucometro_df.drop(labels=excluye, axis=0, inplace=True)


# consulta online a la API para las visitas, que proporcionan la info de
# visitas y microbiota. la función regresa info solo de participantes incluidos
visitas_datos = [ get_visit_data(visita) for visita in range(1, Visitas_e + 1)]
muestras = visitas_datos[0][["id", "muestra"]]
muestras.set_index("id", inplace=True)
for visita in [1, 2, 3]:
    otro = visitas_datos[visita][["id", "muestra"]]
    otro.set_index("id", inplace=True)
    t = muestras.join(otro, lsuffix="l-"+str(visita), how='outer')
    muestras = t
muestras["NSecs"] = muestras.count(axis=1)

num_muestras = [visita["muestra"].count() for visita in visitas_datos]
num_secuencias = [get_secuenciaciones_data(visita) for visita in [1,2,3,4]]


# grafica estática de desempeño de reclutados.
desempenio = px.pie(reclutados, names="finished_periods",  title="Etapas terminadas por la población reclutada", labels={'finished_periods':'Etapas concluidas'})
# gráfica estática de distribución de comidas
comidas_dist = px.pie(comidas_df, names="etapas", title="Comidas por etapa")
external_stylesheets = [
    {
        "href": (
            "https://fonts.googleapis.com/css2?"
            "family=Lato:wght@400;700&display=swap"
        ),
        "rel": "stylesheet",
    },
]

app = Dash(__name__, external_stylesheets=external_stylesheets, server=server, url_base_pathname='/nutri/inventario/')
app.title = "NutrINDmex. Inventario de datos"

app.layout = html.Div(children=[
    #encabezado
    html.Div(children=[
        html.P(children="🥑", className="header-emoji"),
        html.H1(children="Inventario de datos", className="header-title"),
        html.P(children=(str(len(reclutados.index)) + " participantes fueron reclutados."),className="header-description"),
        html.P(children=(str(len(participantes.index)) + " están incluidos en el estudio."),className="header-description"),
    ],
    className="header",),
    html.Div(children=[html.H1("Desempeño de la población reclutada", className="header-title2")]),
    html.Div(children=[dcc.Graph(figure=desempenio)]),
    # cobertura por fuente de datos
    html.Div(children=[ html.H1("Cobertura de pacientes por fuente de datos", className="header-title2") ]),
    # dos columnas: controles y gráfica de pacientes por fuente de datos
    html.Div(children=[
        #controles
        html.Div(children=[
            html.Div(children="Etapa", className="menu-title"),
            dcc.Dropdown(
                id="etapaSel",
                options=[
                    {"label":1, "value":1},
                    {"label":2, "value":2},
                    {"label":3, "value":3},
                ],
                value=1,
                clearable=False,
                searchable=False,
                className="dropdown",
            ),
            html.Div(children="Comidas", className="menu-title"),
            dcc.Slider(
                id="min_comidas",
                min=1,
                max=Comidas_totales_e,
                value=15,
                marks={1:"1",15:"15", 21:"21",30:"30",42:"42", Comidas_totales_e:str(Comidas_totales_e)}
            ),
            html.Div(children="Días con glucómetro", className="menu-title"),                        
            dcc.Slider(
                id="min_glucosa",
                min=1,
                max=Glucometro_e,
                value=7,
                marks={1:"1", 7:"7", 14:"14", 28:"28", 42:"+42"}
            ),
        ]),
        #gráfica
        html.Div(children=[dcc.Graph(id="gauges")],)
    ],style={'display': 'flex', 'flexDirection': 'row'}, className="wrapper"),
    # segundo 'panel': datos existentes por paciente. 
    html.Div(children=[
        html.H1("Cantidad de datos registrados por paciente", className="header-title2"),
        # controles. 1 renglon, dos columnas.
        html.Div(children=[
            #dropdown etapa. columna izquierda.
            html.Div(children=[
                html.Div(children="Etapa", className="menu-title"),
                dcc.Dropdown(
                    id="select_etapa_dpp",
                    options=[
                        {"label":1, "value":1},
                        {"label":2, "value":2},
                        {"label":3, "value":3},
                    ],
                    value=1,
                    clearable=False,
                    searchable=False,
                    className="dropdown",
                ),
            ], className="wrapper"),
            # dropdown fuente. columna derecha.
            html.Div(children=[
                html.Div(children="Fuente", className="menu-title"),
                dcc.Dropdown(
                    id="select_fuente_dpp",
                    options=[
                        {"label":"Comidas", "value":1},
                        {"label":"Glucómetro", "value":2},
                    ],
                    value=1,
                    clearable=False,
                    searchable=False,
                    className="dropdown",
                ), 
            ],className="wrapper",),
        ], style={"display":"flex", "flexDirection": "row"},),
        dcc.Graph(id="datos_por_paciente"),
    ], className="wrapper"),
    # cobertura de pacientes por prueba. working.
    html.Div(children=[
        html.H1("Cobertura de pacientes por protocolo.", className="header-title2"),
        # selector de etapa
        html.Div(children=[
            html.Div(children="Etapa", className="menu-title"),
            dcc.Dropdown(
                id="select_vis_protocolo",
                options=[
                    {"label":1, "value":1},
                    {"label":2, "value":2},
                    {"label":3, "value":3},
                    {"label":4, "value":4}
                ],
                value=1,
                clearable=False,
                searchable=False,
                className="dropdown",
            ),
        ], className="wrapper"),
        #y gráfica
        dcc.Graph(id="cobertura_protocolo"),
    ], className="wrapper"),
    # datos para análisis de comidas
    html.Div(children=[
        html.H1("Datos para Análisis de comidas", className="header-title2"),
        html.P("..en construcción..", style={"color":"red", "font-style":"italic"}),                
        # dropdown
        html.Div(children=[
            html.Div(children="Etapa", className="menu-title"),
            dcc.Dropdown(
                id="mesurableEt",
                options=[
                    {"label":"1", "value":1},
                    {"label":"2", "value":2},
                    {"label":"3", "value":3},
                    {"label":"Total", "value":5},                 
                ],
                value=1,
                clearable=False,
                searchable=False,
                className="dropdown",
            ), 
        ], className="wrapper"),
        dcc.Graph(figure=comidas_dist)                
    ], className="wrapper")
]) # cierre del layout

@app.callback(
    Output("gauges", "figure"),
    Input("etapaSel", "value"),
    Input("min_comidas", "value"),
    Input("min_glucosa", "value"),    
)
def update_gauges(etapaSel, min_comidas, min_glucosa):
    gauges = {
            'shape': "bullet",
            'axis': {'range': [None, 100]},
            'steps': [
                {'range': [0, 25], 'color': "lightcyan"},
                {'range': [25, 50], 'color': "lightblue"},                
                {'range': [50, 75], 'color': "skyblue"},
                {'range': [75, 100], 'color': "steelblue"}],            
            'bar': {'color': "blue"},
        }
    numbers = { 'suffix':'%', 'valueformat':'.2f'}
    modes = "number+gauge"

    comidas_data = resumen_comidas.loc[(slice(None),f'e{etapaSel}')] > min_comidas
    glucosa_data = glucometro_df.loc[glucometro_df["visit_id"] == etapaSel, "amount_of_days"] > min_glucosa
    microbiota_data = num_secuencias[etapaSel-1]

#    microbiota_data = visitas_datos[etapaSel-1]["muestra"].count()
# La N de la etapa debe ser el número de participantes que cumplieron la etapa.
# esto es diferente al número de participantes que asistieron a la visita N o N+1.
    N_etapa = len(participantes.loc[participantes["finished_periods"].isin(range(etapaSel, 4))].index)

    multi = go.Figure()
    multi.add_trace(go.Indicator(
       value = 100 * comidas_data.sum() / N_etapa,
        domain = {'x': [0.25, 1], 'y': [0.60, 0.7]},
        title = {'text': "Comidas"},
        mode = modes,
        number = numbers,
        gauge = gauges,
    ))
    multi.add_trace(go.Indicator(
        value = 100 * glucosa_data.sum() / N_etapa,
        domain = {'x': [0.25, 1], 'y': [0.8, 0.9]},
        title = {'text': "Glucómetro"},
        mode = modes,
        number = numbers,
        gauge = gauges
    ))
    multi.add_trace(go.Indicator(
        value = 100 * microbiota_data / N_etapa,
        domain = {'x': [0.25, 1], 'y': [0.5, 0.6]},
        title = {'text': "Microbiota"},
        mode = modes,
        number = numbers,
        gauge = gauges
    ))
    
    return multi    

@app.callback(
    Output("datos_por_paciente", "figure"),
    Input("select_etapa_dpp", "value"),
    Input("select_fuente_dpp", "value")
)
def update_datos_por_paciente(etapa, fuente):
    if fuente == 1:
        xs = resumen_comidas.loc[(slice(None),f'e{etapa}')].sort_values()
        title = f"Comidas por paciente en la etapa {etapa}"
        medCadena = 'comidas'
    if fuente == 2:
        xs = glucometro_df.loc[glucometro_df["visit_id"] == etapa]["amount_of_days"].sort_values()
        title = f'dias de glucómetro por paciente en la etapa {etapa}'
        medCadena = 'días de glucómetro'

    xs.index = "p_" + xs.index.astype(str)
    figura = {
        "data": [
            {
                "x": xs.index,
                "y": xs,
                "name": "Comidas",
                #"orientation": "v",
            },
            #{ "x": temp["Comidas_e1"], "type": "histogram"},
        ],
        "layout": {
            "title": {"text": title},
            "xaxis": { "title": {"text": "Paciente"}},
            "yaxis": { "title": {"text": f'Número de {medCadena}'}},
            "bargap": "0.05"
        }
    }

    return figura


@app.callback(
    Output("cobertura_protocolo","figure"),
    Input("select_vis_protocolo", "value")
)
def update_por_protocolo(visita):
    lcol = 1
    lrow = 1
    df1 = visitas_datos[visita-1]
    protocolos = {
        "pyhsiological" : ["pas", "pad", "ipaq", "stress", "hadsa", "hadsd"],
        "tolerance_curves" : ["Pan", "Gelatina", "Glucosa"],
        "corporal_composition" : ["mmkg", "mmekg", "mgkg", "mlgkg", "percent_mg", "gv"],
        "biochemical" :["hba1c", "glucose", "ct", "tag", "ldl", "hdl", "pcr","alt", "ast","homa", "insuline" ],
        "antropometric" : ["imc", "weight", "waist_circumference"]
    }
    figura = make_subplots(rows=2, cols=3, vertical_spacing=0.25)
    for protocolo in protocolos.keys():
        columnas = list(set(df1.columns).intersection(set(protocolos[protocolo])))
        cuentas = 100 * df1[columnas].replace("",np.nan).count() / len(df1)
        figura.add_trace(
            go.Bar(x=cuentas.index, y=cuentas),
            row = lrow,
            col = lcol,
        )
        lcol = lcol + 1
        if lcol > 3:
            lcol = 1
            lrow = lrow + 1
    return figura


if __name__ == "__main__":
    app.run_server(debug=True, port=8090, host='0.0.0.0')
