import pandas as pd
from dash import Dash, Input, Output, dcc, html
import plotly.graph_objects as go
import plotly.express as px
import requests
import numpy as np
import flask

server = flask.Flask(__name__)

# algunos parámetros globales
url = 'https://nutricion.c3.unam.mx/nd/'
Visitas_e = 4
Etapas_e = 3
Comidas_E1_e = 42
Comidas_E2_e = 9
Comidas_E3_e = 9
#Comidas_totales_e = Comidas_E1_e + Comidas_E2_e + Comidas_E3_e
Comidas_totales_e = 42
Glucometro_e = 14

esperados = {
    "antrop-mets": 3,
    "quimica-S": 13,
    "comp-corp":  6,
    "physiological": 10,
    "tolerancia": 3
    
}

def get_visit_data(visitnumber):
    url_visitas = url + "/visits/" + str(visitnumber) + "/patients-data"
    yaisons = requests.get(url_visitas).json()["patient_data"]
    df = pd.json_normalize(yaisons)
    df.drop(["tolerance_curve_measure","name","patient_visit_id", "visit_date"], inplace=True, axis=1)
    lista = [ { cf["controlled_food"]: cf["increment_auc"] for cf in yaison["tolerance_curve_measure"] } for yaison in yaisons ] 
    df.rename(columns= lambda x: x.split(".")[1] if "." in x else x, inplace=True)
    #df.drop(["patient_visit_id"], inplace=True, axis=0) #esta etiqueta aparece dos veces: dentro del primer nivel de json y dentro del json de sample
    return pd.concat([df, pd.DataFrame(lista)], axis=1)

# fuentes de datos

# consulta online a la API para las visitas, que proporcionan la info de
# visitas y microbiota
visitas_datos = [ get_visit_data(visita) for visita in range(1, Visitas_e + 1)]
# la info de glucómetro y comidas es más compleja. por el momento, se estab armando
# la info offline.

data_o = (
    pd.read_csv("inventario.csv")
)

data_o["paciente"] = data_o["paciente"].astype("str") + "_p"
data_o["Secuenciaciones"] = 0

Total_pacientes = len(data_o.index)
data = data_o.loc[data_o["incluido"]]
Total_incluidos = len(data)

secuencias = visitas_datos[0][["id", "patient_visit_id"]]
secuencias.set_index("id", inplace=True)
for visita in [1, 2, 3]:
    otro = visitas_datos[visita][["id", "patient_visit_id"]]
    otro.set_index("id", inplace=True)
    t = secuencias.join(otro, lsuffix="l-"+str(visita), how='outer')
    secuencias = t
secuencias["NSecs"] = secuencias.count(axis=1)
                          


# grafica estática de visitas/secuenciaciones
scat = px.scatter(data, x="NDias_glucosa", y="Comidas_Aisladas",size="AUCs_glucosa")
vis_sec = {
        "data": [
            {
                #"x": temp["paciente"],
                "x": data["Nvisitas"].astype("str") + "_visitas",
                "type": "histogram",
                "name": "Visitas",
                "histnorm": "percent",                
                #"orientation": "v",
            },
            { "x": secuencias["NSecs"], "type": "histogram", "name": "Secuenciaciones", "histnorm": "percent"},
        ],
        "layout": {
            #"title": {"text": title + " " + etapalab + str(etapaDet)},
            "xaxis": { "title": {"text": "Visitas por paciente"}},
            "yaxis": { "title": {"text": "% de pacientes"}},            
        }
    }


external_stylesheets = [
    {
        "href": (
            "https://fonts.googleapis.com/css2?"
            "family=Lato:wght@400;700&display=swap"
        ),
        "rel": "stylesheet",
    },
]

app = Dash(__name__, external_stylesheets=external_stylesheets, server=server, url_base_pathname='/nutri/inventario/')
app.title = "NutrINDmex. Inventario de datos"

app.layout = html.Div(children=[
    #encabezado
    html.Div(children=[
        html.P(children="🥑", className="header-emoji"),
        html.H1(children="Inventario de datos", className="header-title"),
        html.P(children=(str(Total_pacientes) + " participantes."),className="header-description"),
        html.P(children=(str(Total_incluidos) + " están incluidos en el análisis."),className="header-description"),
    ],
    className="header",),
    # cobertura por fuente de datos
    html.Div(children=[ html.H1("Cobertura de pacientes por fuente de datos", className="header-title2") ]),
    # dos columnas: controles y gráfica de pacientes por fuente de datos
    html.Div(children=[
        #controles
        html.Div(children=[
            html.Div(children="Etapa", className="menu-title"),
            dcc.Dropdown(
                id="etapaSel",
                options=[
                    {"label":1, "value":1},
                    {"label":2, "value":2},
                    {"label":3, "value":3},
                ],
                value=1,
                clearable=False,
                searchable=False,
                className="dropdown",
            ),
            html.Div(children="Comidas", className="menu-title"),
            dcc.Slider(
                id="min_comidas",
                min=1,
                max=Comidas_totales_e,
                value=15,
                marks={1:"1",15:"15", 21:"21",30:"30",42:"42", Comidas_totales_e:str(Comidas_totales_e)}
            ),
            html.Div(children="Días con glucómetro", className="menu-title"),                        
            dcc.Slider(
                id="min_glucosa",
                min=1,
                max=Glucometro_e,
                value=7,
                marks={1:"1", 7:"7", 14:"14", 28:"28", 42:"+42"}
            ),
        ]),
        #gráfica
        html.Div(children=[dcc.Graph(id="multi-chart")],)
    ],style={'display': 'flex', 'flexDirection': 'row'}, className="wrapper"),
    # segundo 'panel': datos existentes por paciente.
    html.Div(children=[
        html.H1("Cantidad de datos por paciente", className="header-title2"),
        # controles. 1 renglon, dos columnas.
        html.Div(children=[
            #dropdown etapa. columna izquierda.
            html.Div(children=[
                html.Div(children="Etapa", className="menu-title"),
                dcc.Dropdown(
                    id="etapaDet",
                    options=[
                        {"label":1, "value":1},
                        {"label":2, "value":2},
                        {"label":3, "value":3},
                        {"label":"No-Clas.", "value":99}
                    ],
                    value=1,
                    clearable=False,
                    searchable=False,
                    className="dropdown",
                ),
            ], className="wrapper"),
            # dropdown fuente. columna derecha.
            html.Div(children=[
                html.Div(children="Fuente", className="menu-title"),
                dcc.Dropdown(
                    id="medidaDet",
                    options=[
                        {"label":"Comidas", "value":1},
                        {"label":"Glucómetro", "value":2},
                    ],
                    value=1,
                    clearable=False,
                    searchable=False,
                    className="dropdown",
                ), 
            ],className="wrapper",),
        ], style={"display":"flex", "flexDirection": "row"},),
        dcc.Graph(id="detalle_figura"),
    ], className="wrapper"),
    # gráficas de visitas
    html.Div(children=[
        html.H1("Visitas / Secuenciaciones", className="header-title2"),
        html.Div(children=[
                dcc.Graph(figure=vis_sec),
        ], className="wrapper"),
    ]),
    # cobertura de visitas
    html.Div(children=[
        html.H1("Detalle de visitas", className="header-title2"),
        html.P("..en construcción..", style={"color":"red", "font-style":"italic"}),
        html.Div(children="Visita", className="menu-title"),
        dcc.Dropdown(
            id="visitaSel",
            options=[
                {"label":1, "value":1},
                {"label":2, "value":2},
                {"label":3, "value":3},
                {"label":4, "value":4}
            ],
            value=1,
            clearable=False,
            searchable=False,
            className="dropdown",
        ),
        dcc.Graph(id="multi-visitas"),
    ], className="wrapper",),
    # cobertura de pacientes por prueba
    html.Div(children=[
        html.H1("Cobertura de pacientes por prueba", className="header-title2"),
        # selector de etapa
        html.Div(children=[
            html.Div(children="Etapa", className="menu-title"),
            dcc.Dropdown(
                id="visitaPruebas",
                options=[
                    {"label":1, "value":1},
                    {"label":2, "value":2},
                    {"label":3, "value":3},
                    {"label":4, "value":4}
                ],
                value=1,
                clearable=False,
                searchable=False,
                className="dropdown",
            ),
        ], className="wrapper"),
        #y gráfica
        dcc.Graph(id="detalle_pruebas"),
    ], className="wrapper"),
    # datos para análisis de comidas
    html.Div(children=[
        html.H1("Datos para Análisis de comidas", className="header-title2"),
        html.P("..en construcción..", style={"color":"red", "font-style":"italic"}),                
        # dropdown
        html.Div(children=[
            html.Div(children="Etapa", className="menu-title"),
            dcc.Dropdown(
                id="mesurableEt",
                options=[
                    {"label":"1", "value":1},
                    {"label":"2", "value":2},
                    {"label":"3", "value":3},
                    {"label":"Total", "value":5},                 
                ],
                value=1,
                clearable=False,
                searchable=False,
                className="dropdown",
            ), 
        ], className="wrapper"),
        dcc.Graph(id="mesurables")                
    ], className="wrapper")
]) # cierre del layout


@app.callback(
    Output("multi-chart", "figure"),
    Output("detalle_figura", "figure"),
    Output("multi-visitas", "figure"),
    Output("detalle_pruebas", "figure"),
    Output("mesurables", "figure"),                
    Input("etapaSel", "value"),
    Input("min_comidas", "value"),
    Input("min_glucosa", "value"),
    Input("medidaDet", "value"),
    Input("etapaDet", "value"),
    Input("visitaSel", "value"),
    Input("visitaPruebas", "value"),
    Input("mesurableEt", "value"),            

)
def update_charts(etapaSel,min_comidas, min_glucosa, medidaDet, etapaDet, visitaSel, visitaPruebas, mesurableEt):
    contados = {}
    total_data = len(data.index)
    gauges = {
            'shape': "bullet",
            'axis': {'range': [None, 100]},
            'steps': [
                {'range': [0, 25], 'color': "lightcyan"},
                {'range': [25, 50], 'color': "lightblue"},                
                {'range': [50, 75], 'color': "skyblue"},
                {'range': [75, 100], 'color': "steelblue"}],            
            'bar': {'color': "blue"},
        }
    numbers = { 'suffix':'%', 'valueformat':'.2f'}
    modes = "number+gauge"

    comidas_data = pd.DataFrame()
    glucosa_data = pd.DataFrame()
    microbiota_data = pd.DataFrame()
    fismet_data = pd.DataFrame()
    qs_data = pd.DataFrame()
    visitas_data = pd.DataFrame()
    
    
    if etapaSel == 1:
        visitas_data = data.loc[data["visita1"].notna()]
        glucosa_data = data.query("NDias_glucosa_e1 >= @min_glucosa")
        comidas_data = data.query("Comidas_e1 >= @min_comidas")
    if etapaSel == 2:
        visitas_data = data.loc[data["visita2"].notna()]
        glucosa_data = data.query("NDias_glucosa_e2 >= @min_glucosa")
        comidas_data = data.query("Comidas_e2 >= @min_comidas")
    if etapaSel == 3:
        visitas_data = data.loc[data["visita3"].notna()]
        glucosa_data = data.query("NDias_glucosa_e3 >= @min_glucosa")        
        comidas_data = data.query("Comidas_e3 >= @min_comidas")        
    if etapaSel == 99:
        comidas_data = data.query("Selectedfoods >= @min_comidas")
        glucosa_data = data.query("NDias_glucosa >= @min_glucosa")
        visitas_data = data.query("Nvisitas >= 4") 

# el API regresa en los datos de visitas dos campos que se llaman patient_visit_id
# uno está en el nivel más externo de la respuestas, y es desechado en la función get_visit_data
# el otro está en un segundo nivel, dentro de la estructura "sample" y sube al primer nivel
# mediante la función json_normalize y después el rename lo convierte en un simple patient_visit_id.

# La razón por la que se desecha el dato del nivel exterior y se conserva el interior es
# que todos los pacientes tienen el primero pero solo los que se hicieron muestra para microbiota tienen el segundo. 

    microbiota_data = visitas_datos[etapaSel-1]["patient_visit_id"].count()
    multi = go.Figure()



    multi.add_trace(go.Indicator(
        value = 100 * len(comidas_data.index) / Total_incluidos,
        domain = {'x': [0.25, 1], 'y': [0.60, 0.7]},
        title = {'text': "Comidas"},
        mode = modes,
        number = numbers,
        gauge = gauges,
    ))


    multi.add_trace(go.Indicator(
        value = 100 * len(glucosa_data.index) / Total_incluidos,
        domain = {'x': [0.25, 1], 'y': [0.8, 0.9]},
        title = {'text': "Glucómetro"},
        mode = modes,
        number = numbers,
        gauge = gauges
    ))

 
    

    multi.add_trace(go.Indicator(
        value = 100 * microbiota_data / Total_incluidos,
#        value = min_microbiota,
        domain = {'x': [0.25, 1], 'y': [0.5, 0.6]},
        title = {'text': "Microbiota"},
        mode = modes,
        number = numbers,
        gauge = gauges
    ))



    multi.add_trace(go.Indicator(
        value = 100 * len(visitas_data.index) / Total_incluidos,
        domain = {'x': [0.25, 1], 'y':[0.9, 1.0]},
        title = {'text': "Visitas"},
        mode = modes,
        number = numbers,
        gauge = gauges
    ))    

    if etapaDet == 99:
        etapaDet = "out"
    else:
        etapaDet = "e" + str(etapaDet)

    etapalab = "etapa "    
    if medidaDet == 1:
        medCadena = "Comidas_"
        title = "Comidas"
    if medidaDet == 2:
        medCadena = "NDias_glucosa_"
        title = "Glucómetro"
    if medidaDet == 4:
        medCadena = "visita"
        etapalab = " "
        title = "Visitas"
        if etapaDet == "out":
            etapaDet = "4"
        else:
            etapaDet = etapaDet.replace('e','')


    medCadena = medCadena + etapaDet
    temp = data.sort_values(medCadena)

    detalle_figura = {
        "data": [
            {
                "x": temp["paciente"],
                "y": temp[medCadena],
                #"type": "histogram",
                #"xticks": temp["paciente"],
                "histnorm": "percent",
                "name": "Comidas",
                #"orientation": "v",
            },
            #{ "x": temp["Comidas_e1"], "type": "histogram"},
        ],
        "layout": {
            #"title": {"text": title + " " + etapalab + str(etapaDet)},
            "xaxis": { "title": {"text": "rango de " + medCadena}},
            "yaxis": { "title": {"text": "% de pacientes"}},
            "bargap": "0.05"
        }
    }


    

    multiV = go.Figure()

    for medicion in esperados.keys():
        contados[medicion] = len(data.loc[data[medicion + str(visitaSel)] > 0])
    
    multiV.add_trace(go.Indicator(
        value = 100 * contados["antrop-mets"] / Total_incluidos,
        domain = {'x': [0.25, 1], 'y': [0.90, 1]},
        title = {'text': "Antropometría"},
        mode = modes,
        number = numbers,
        gauge = gauges,
    ))

    multiV.add_trace(go.Indicator(
        value = 100 * contados["quimica-S"] / Total_incluidos,
        domain = {'x': [0.25, 1], 'y': [0.80, 0.9]},
        title = {'text': "QS"},
        mode = modes,
        number = numbers,
        gauge = gauges,
    ))

    multiV.add_trace(go.Indicator(
        value = 100 * contados["physiological"] / Total_incluidos,
        domain = {'x': [0.25, 1], 'y': [0.7, 0.8]},
        title = {'text': "Fisiológicos"},
        mode = modes,
        number = numbers,
        gauge = gauges,
    ))

    multiV.add_trace(go.Indicator(
        value = 100 * contados["comp-corp"] / Total_incluidos,
        domain = {'x': [0.25, 1], 'y': [0.6, 0.7]},
        title = {'text': "C.Corporal"},
        mode = modes,
        number = numbers,
        gauge = gauges,
    ))

    multiV.add_trace(go.Indicator(
        value = 100 * contados["tolerancia"] / Total_incluidos,
        domain = {'x': [0.25, 1], 'y': [0.5, 0.6]},
        title = {'text': "Curvas de Tolerancia"},
        mode = modes,
        number = numbers,
        gauge = gauges,
    ))    

    url_visitas = url + "/visits/" + str(visitaPruebas) + "/patients-data/"
    t1 = requests.get(url_visitas).json()["patient_data"]
    df = pd.json_normalize(t1)
    df.drop(["tolerance_curve_measure","id","name","finished_periods","include_in_analysis","patient_visit_id", "visit_date","sample"], inplace=True, axis=1)
    l1 = [ { cf["controlled_food"]: cf["increment_auc"] for cf in tolerance["tolerance_curve_measure"] } for tolerance in t1 ] 
    df.rename(columns= lambda x: x.split(".")[1] if "." in x else x, inplace=True)
    df1 = pd.concat([df, pd.DataFrame(l1)], axis=1)
#    df1 = df
    cuentas = 100 *  df1.replace("",np.nan).count() / len(df1)
    
    detalle_pruebas = {
        "data": [
            {
                "x": cuentas.index,
                "y": cuentas,
                "type": "bar",
            }
        ],
    }
    

    por_pasos = 1
    por_rcard = 1
    if mesurableEt < 4:
        columna = "AUCs_glucosa_e" + str(mesurableEt)
        por_glucosa = data[columna].sum()
    if mesurableEt == 4:
        por_glucosa = data["AUCs_glucosa_out"].sum()
    if mesurableEt == 5:
        por_glucosa = data["AUCs_glucosa_e1"].sum() + \
                      data["AUCs_glucosa_e2"].sum() + \
                      data["AUCs_glucosa_e3"].sum() + \
                      data["AUCs_glucosa_out"].sum()
    datos_mesurables = [ por_glucosa, por_pasos, por_rcard]
    numcomidas = data[["AUCs_glucosa_e1","AUCs_glucosa_e2","AUCs_glucosa_e3"]].replace("",np.nan).sum()
    numpacientes = data[["AUCs_glucosa_e1","AUCs_glucosa_e2","AUCs_glucosa_e3"]].replace("",np.nan).count()
    
    mesurables = {
        "data": [
            {
                "x": ["e1", "e2", "e3"],
                "y": numcomidas,
                "type": "bar",
            }
        ],
        "layout": {
            "title": {"text": "Comidas Analizables"}
        }
    }
    
    return   multi, detalle_figura, multiV, detalle_pruebas, mesurables


if __name__ == "__main__":
    app.run_server(debug=True, port=8090, host='0.0.0.0')
