import pandas as pd
from dash import Dash, Input, Output, State, dcc, html, callback
import plotly.graph_objects as go
import plotly.express as px
import plotly.figure_factory as ff
import requests
import numpy as np
from plotly.subplots import make_subplots
from sklearn.cluster import KMeans
from scipy import stats
from ast import literal_eval
from funciones_comidas import readGlucosa
from funciones_comidas import readVisitas
from nutri_utils import setTipo
from nutri_utils import dominantes
import test_ch_2_b

import flask

server = flask.Flask(__name__)


def update_correlacion_macro_iauc():

    datos = comidas.sort_values(by="iAUC", ascending=True)
    datos = datos[np.abs(stats.zscore(datos["iAUC"], nan_policy='omit'))<3]
    datos.reset_index(inplace=True)
    metricas = ["hc_total", "lipids_total","protein_total","fiber_total"]
    pcts = [ metrica + "_pct" for metrica in metricas]
    datos["total"] = datos[metricas].sum(axis=1)
    datos = datos[datos["total"]>0]
    datos[pcts] = datos[metricas].div(datos["total"], axis=0)
    
    fig = make_subplots(rows=2, cols=4, vertical_spacing=0.25)
    lcol = 1
    seq = 1
    for metrica in metricas:
        datosfil = datos[np.abs(stats.zscore(datos[metrica], nan_policy='omit')) < 3]     

        fig.add_trace(
            go.Scatter(x=datosfil["iAUC"], y=datosfil[metrica], name=metrica, mode='markers'),
            row=1, col=lcol
        )
        fig.add_trace(
            go.Scatter(x=datosfil["iAUC"], y=datosfil[metrica + "_pct"], name=metrica, mode='markers'),
            row=2, col=lcol
        )
        fig["layout"]['xaxis'  + str(seq)]['title']="iAUC" 
        fig["layout"]['yaxis'  + str(seq)]['title']= metrica
        fig["layout"]['xaxis'  + str(seq + 4)]['title']="iAUC" 
        fig["layout"]['yaxis'  + str(seq + 4)]['title']= "pct de " + metrica

        lcol = lcol + 1
        seq = seq + 1
    return fig

def comidas_iaucs(datos):
    columnas = ["pre-AUC", "post-AUC", "iAUC"]
    datos["N-alimentos"] = datos["foods"].apply(lambda x: len(x))

    fig = make_subplots(cols=3, rows=1,  vertical_spacing=0.25) 
    lcol = 1
    seq = 1
    for columna in columnas:
        datosfil = datos[np.abs(stats.zscore(datos[columna], nan_policy='omit')) < 3]
        fig.add_trace(
            go.Scatter(y=datosfil[columna], name=columna, mode="markers"),
            row=1, col=lcol
        )
        fig['layout']['yaxis' + str(seq)]['title']=columna
        fig['layout']['xaxis' + str(seq)]['title']="Comida ID"                        
        seq = seq + 1
        lcol = lcol + 1
    fig.update_layout(showlegend=False)
    return fig

def comidas_iauc_describe(datos):
    columnas = ["pre-AUC", "post-AUC", "iAUC"]
    fig = make_subplots(cols=3, rows=1, vertical_spacing=0.25)
    seq = 1
    lcol = 1
    for columna in columnas:
        datosfil = datos[np.abs(stats.zscore(datos[columna], nan_policy='omit')) < 3]
        fig.add_trace(
            go.Histogram(x=datosfil[columna], name=columna),
            row=1, col=lcol
        )
        fig['layout']['yaxis' + str(seq)]['title']=columna
        seq = seq + 1
        lcol = lcol + 1 


    fig.update_layout(
        showlegend = False,
        bargap=0.05, # gap between bars of adjacent location coordinates
        bargroupgap=0.05 # gap between bars of the same location coordinates
    )
    return fig

def comidas_macro_contenido(datos):
    columnas = ["hc_total", "lipids_total", "fiber_total", "protein_total"]
    datos["N-alimentos"] = datos["foods"].apply(lambda x: len(x))

    figura = make_subplots(cols=4, rows=1,  vertical_spacing=0.24)
    lcol = 1
    seq = 1
    for columna in columnas:
        datosfil = datos[np.abs(stats.zscore(datos[columna], nan_policy='omit')) < 3]
        figura.add_trace(
            go.Scatter(y=datosfil[columna], name=columna, mode="markers"),
            row=1, col=lcol
        )
        figura['layout']['yaxis' + str(seq)]['title']=columna
        figura['layout']['xaxis' + str(seq)]['title']="Comida ID"                        
        seq = seq + 1
        lcol = lcol + 1 if lcol < 4 else 1
    figura.update_layout(showlegend=False)
    return figura

def comidas_macro_describe(datos):
    columnas = ["hc_total", "lipids_total", "fiber_total", "protein_total"]
    fig = make_subplots(cols=4, rows=1, vertical_spacing=0.25)
    seq = 1
    lcol = 1
    for columna in columnas:
        datosfil = datos[np.abs(stats.zscore(datos[columna], nan_policy='omit')) < 3]
        fig.add_trace(
            go.Histogram(x=datosfil[columna], name=columna),
            row=1, col=lcol
        )
        fig['layout']['yaxis' + str(seq)]['title']=columna
        seq = seq + 1
        lcol = lcol + 1 if lcol < 4 else 1


    fig.update_layout(
        showlegend = False,
        bargap=0.05, # gap between bars of adjacent location coordinates
        bargroupgap=0.05 # gap between bars of the same location coordinates
    )
    return fig

def comidas_porcentajes(datos):
    columnas = ["hc_total", "fiber_total", "protein_total", "lipids_total"]
    datos["total"] = datos[columnas].sum(axis=1)
    datos = datos[datos["total"]>0]
    datos[columnas] = datos[columnas].div(datos["total"], axis=0)
    
    figura = ff.create_distplot([ datos[columna] for columna in columnas], columnas, show_hist = False, show_rug=False)
    return figura

def comidas_principales(datos):
    dominantes = datos.groupby("dominante").count()
    dominantes.sort_values(by="patient", ascending=False, inplace=True)
    dominantes = dominantes[dominantes["patient"] > 9]
    figura = make_subplots(cols=1, rows=1, subplot_titles= ["Alimentos"])
    figura.add_trace(
        go.Scatter(x=dominantes.index, y=dominantes["patient"], mode='markers', name="# de comidas"),
        col=1, row=1
    )
    figura["layout"]['yaxis']['title']= "# de comidas"
#    figura.update_layout(height=2000, width=800)    
    return figura

def comidas_principales_categorias(datos):
    categorias = datos.groupby("dominante_category").count()
    categorias.sort_values(by="patient", ascending=False, inplace=True)
    categorias = categorias[categorias["patient"] > 9]    
    figura = make_subplots(cols=1, rows=1, subplot_titles=["Categorías de alimentos"])
    figura.add_trace(
        go.Scatter(x=categorias.index, y=categorias["patient"], mode='markers'),
        col=1, row=1
    )
    figura["layout"]['yaxis']['title']= "# de comidas"

    return figura

def categorias_promedio_aucs():

    categorias = comidas.groupby("dominante_category")[["iAUC","hc_total"]].mean()
    #print(categorias)
    categorias["tamaño"] = comidas.groupby("dominante_category")["iAUC"].count()
    categorias.sort_values(ascending=False, inplace=True, by="iAUC")

    # figura = make_subplots(cols=1, rows=1, subplot_titles=["Categorías de alimentos"])
    # figura.add_trace(
    #     go.Scatter(x=categorias.index, y=categorias["iAUC"], mode="markers")
    # )
    figura = px.scatter(categorias["iAUC"], size=categorias["tamaño"])
    return figura

def categorias_aucs(datos):
    datos_fil = datos[np.abs(stats.zscore(datos["iAUC"], nan_policy='omit')) < 3]
    figura = px.box(datos_fil, x="dominante_category", y="iAUC")

    return figura

def aucs_paciente(datos):

    figura = px.box(datos, x="patient", y="iAUC")
    return figura

def comidas_paciente_datos(datos):
    columnas = ["hc_total", "kcal_total", "fiber_total", "protein_total", "lipids_total"]
    figura = make_subplots(cols=5, rows=4, subplot_titles=("hc","kcal","fiber", "protein", "lipids"))
    lcol = 1
    for columna in columnas:
        datosfil = datos[np.abs(stats.zscore(datos[columna], nan_policy='omit')) < 3]        
        figura.add_trace(
            go.Scatter(y=datosfil.groupby("patient")[columna].sum(), name="total_" + columna, mode= "markers"),
            row=1, col=lcol
        )
        figura.add_trace(
            go.Histogram(x=datosfil.groupby("patient")[columna].sum(), name="total_" + columna),
            row=2, col=lcol
        )

        figura.add_trace(
            go.Scatter(y=datosfil.groupby("patient")[columna].mean(), name="promedio_" + columna, mode= "markers"),
            row=3, col=lcol
        )
        figura.add_trace(
            go.Histogram(x=datosfil.groupby("patient")[columna].mean(), name="promedio_" + columna),
            row=4, col=lcol
        )
        lcol = lcol + 1
    figura.update_layout(showlegend = False)
    return figura    

def comidas_auc_paciente(datos):
    columnas = ["pre-AUC", "post-AUC", "iAUC"]
    figura = make_subplots(cols = 4, rows = 2, subplot_titles=("pre-AUC","post-AUC", "iAUC", "N-Comidas"))
    lcol = 1
    for columna in columnas:
        datosfil = datos[np.abs(stats.zscore(datos[columna], nan_policy='omit')) < 3]        
        figura.add_trace(
            go.Scatter(y=datosfil.groupby("patient")[columna].mean(), name="promedio_" + columna, mode= "markers"),
            row=1, col=lcol
        )
        figura.add_trace(
            go.Histogram(x=datosfil.groupby("patient")[columna].mean(), name="Distr. " + columna),
            row=2, col=lcol
        )
        lcol = lcol + 1
    figura.add_trace(
        go.Scatter(y=datosfil.groupby("patient")["iAUC"].count(), name="N-comidas", mode= "markers"),
        row = 1, col = 4
    )
    figura.add_trace(
        go.Histogram(x=datosfil.groupby("patient")["iAUC"].count(), name="Distr. N-comidas"),
        row = 2, col = 4
    )
    figura.update_layout(
        showlegend = False,
        bargap=0.05, # gap between bars of adjacent location coordinates
        bargroupgap=0.05 # gap between bars of the same location coordinates
    )

    return figura 

df_total = test_ch_2_b.foods()
# hay dos  gráficas exploratorias que usan el contenido de 'foods'.
# hay que revisar los foods de las foods que vienen de los recordotorios -no se dejan convertir de string a list-, y ver que hacer
# con las comidas de  WA -porque no tienen foods todavía.
# mientras, trabajamos las estadísticas solo con las que vienen de la APP.
df_estadisticas = df_total.loc[df_total["Source"] == "app"]
df_estadisticas["foods"] = df_estadisticas["foods"].apply(literal_eval)
first_id = df_total["patient"].min()
last_id = df_total["patient"].max()
ids_presentes = set(df_total['patient'])
df_total['FH_salto'] = pd.to_datetime(df_total['FH_salto'])
df_total['FH_reportada'] = pd.to_datetime(df_total['FH_reportada'])
df_total['FH_foto'] = pd.to_datetime(df_total['FH_foto'])
# convertimos las comidas de df_estadisticas a un formato que entienden las gráficas
# de estadisticas .. luego, si es conveniente, iremos haciendo los cambios para usar un
# único DF para todas las gráficas. (de todos modos, el entrenamiento de modelos usa
# archivos en el formato en cuestión)
comidas = setTipo( "reportada", df_estadisticas.copy())
comidas[["foods", "Source"]] = df_estadisticas[["foods","Source"]]
comidas.rename(columns={"glucosa_basal_area":"pre-AUC", "glucosa_estimulo_area":"post-AUC", "glucosa_area_efectiva":"iAUC"}, inplace=True)
comidas["iAUC"] = comidas["post-AUC"] - comidas["pre-AUC"] * 2
resultado = pd.DataFrame()
resultado = pd.DataFrame(comidas["foods"].apply(dominantes).values.tolist())
resultado.index = comidas.index
comidas = pd.concat([comidas, resultado], axis=1)

external_stylesheets = [
    {
        "href": (
            "https://fonts.googleapis.com/css2?"
            "family=Lato:wght@400;700&display=swap"
        ),
        "rel": "stylesheet",
    },
]

app = Dash(__name__, external_stylesheets=external_stylesheets, server=server, url_base_pathname='/nutri/estadisticas/')
app.title = "NutrIndMex. Análisis de datos de las comidas."

app.layout = html.Div(children=[
    #encabezado
    html.Div(children=[
        html.P(children="🥑", className="header-emoji"),
        html.H1(children="NutrIndMex", className="header-title"),
        html.H2(children="Análisis de Datos de las Comidas", className="header-title"),

    ],
    className="header",),
    html.Div(children=[
        html.Div(children= [ html.H1("Glucosa de los pacientes con Comidas", className= "header-title2"),],
                 ),
    ]),
    
    dcc.Dropdown(
        options=[
            {'label': 'Todas las comidas', 'value': '0'},
            {'label': '[0-5]', 'value': '[0-5]'},
            {'label': '[5-15]', 'value': '[5-15]'},
            {'label': '[15-30]', 'value': '[15-30]'},
            {'label': '[30-45]', 'value': '[30-45]'},
            {'label': '[45-60]', 'value': '[45-60]'},
            {'label': '[60-90]', 'value': '[60-90]'},
            {'label': '[90-120]', 'value': '[90-120]'},
            {'label': '[>120]', 'value': '[>120]'},
        ],
        value='0', 
        id='rango'
    ),
    dcc.Graph(id='glucosa-graph'),
    dcc.Slider(
        min=first_id, max=last_id, step=1, marks={i: str(i) for i in range(first_id, last_id, 18)},
        value=first_id, tooltip={'always_visible': True}, updatemode='drag', id='slider-patient'
    ),
    html.Div(children= [
        html.Div(children= [html.H1("iAUCs de las comidas reportadas", className = "header-title2")]),
        dcc.Graph(figure=comidas_iaucs(comidas)),
        dcc.Graph(figure=comidas_iauc_describe(comidas))
    ,]),
    # datos de comidas
    html.Div(children= [
        html.Div(children= [html.H1("Contenido de macronutrientes de las comidas", className = "header-title2")]),
        dcc.Graph(figure=comidas_macro_contenido(comidas))
    ,]),

    # descripcion de comidas
    html.Div(children= [
        html.Div(children= [html.H1("Distribución de Macronutrientes de comidas", className = "header-title2")]),
        dcc.Graph(figure=comidas_macro_describe(comidas))
    ,]),
    # descripcion de componentes de macronutrientes de comidas
    html.Div(children= [
        html.Div(children= [html.H1("Distribución de %s de Macronutrientes de comidas", className = "header-title2")]),
        dcc.Graph(figure=comidas_porcentajes(comidas))
    ,]),
    # relación  de  macronutrientes de comidas con sus iAUC
    html.Div(children= [
        html.Div(children= [html.H1("Relación de macros / iAUCs de las comidas ", className = "header-title2")]),
        dcc.Graph(figure=update_correlacion_macro_iauc())
    ,]),

    # # de comidas por componente dominante
    html.Div(children= [
        html.Div(children= [html.H1("Distribución de num. de comidas  por componente principal", className = "header-title2")]),
        dcc.Graph(figure=comidas_principales(comidas)),
        dcc.Graph(figure=comidas_principales_categorias(comidas)),
        dcc.Graph(figure=categorias_promedio_aucs()),
        dcc.Graph(figure=categorias_aucs(comidas))            
    ,]),
    
    # descripcion de comidas por paciente
    html.Div(children= [
        html.Div(children= [html.H1("Macronutrientes de comidas por paciente", className = "header-title2")]),
        dcc.Graph(figure=comidas_paciente_datos(comidas))
    ,]),
    # descripcion de aucs comidas por paciente
    html.Div(children= [
        html.Div(children= [html.H1("AUCs promedio de comidas por paciente", className = "header-title2")]),
        dcc.Graph(figure=comidas_auc_paciente(comidas))
    ,]),
    # descripcion de aucs comidas por paciente
    html.Div(children= [
        html.Div(children= [html.H1("iAUCs de comidas por paciente", className = "header-title2")]),
        dcc.Graph(figure=aucs_paciente(comidas))
    ,]),
    
    
]) # cierre del layout

@callback(
    Output('glucosa-graph', 'figure'),
    [Input('slider-patient', 'value'),
     Input('rango', 'value')],
)
def update_graph(selected_pat, rango_str):
    if selected_pat not in ids_presentes:
        return go.Figure()
    if not selected_pat or not rango_str:
        return {} 
    
    patient_foods = df_total.loc[df_total['patient'] == selected_pat]
    patient_foods_app = patient_foods[patient_foods["Source"] == "app"]
    patient_foods_Wapp = patient_foods[patient_foods["Source"] == "WA"]
    patient_foods_recalls = patient_foods[patient_foods["Source"] == "Recalls"]    
    glucosa = readGlucosa(selected_pat)
    #visita = readVisitas(1)
    #fecha_visita1 = pd.to_datetime(visita.loc[visita['id'] == selected_pat, 'visit_date'].iloc[0])
    if glucosa.empty:
        return go.Figure() 
    #filtered_glucosa = glucosa[(glucosa['date_time'] >= fecha_visita1) & 
     #                          (glucosa['date_time'] <= fecha_visita1 + pd.Timedelta(days=15))]
    
    # Si no hay datos de glucosa en el rango de tiempo, retornar un gráfico vacío
    #if filtered_glucosa.empty:
     #   return go.Figure() 
    fig = go.Figure()
    fig.add_trace(go.Scatter(x=glucosa['date_time'],    # gráfica de la glucosa del paciente seleccionado
                             y=glucosa['value'], 
                             mode='lines', 
                             name='Glucosa del paciente'))
    fig.update_layout(yaxis=dict(range=[50,200]))
    y_foods = np.interp(patient_foods_app['FH_reportada'], (glucosa['date_time']), glucosa['value']) # interpolar fecha de las comidas con glucosa
    distancias = abs((patient_foods['FH_salto'] - patient_foods['FH_reportada']).dt.total_seconds()) / 60
    distancias_df = pd.concat([distancias,patient_foods['FH_reportada'], patient_foods['FH_foto']], axis=1, ignore_index=True)
    distancias_df.columns = ['Distancia','FH_reportada', 'FH_foto']

    y_foto = np.interp(pd.to_numeric(patient_foods_app['FH_foto']), pd.to_numeric(glucosa['date_time']),  # interpolar fecha de la foto con glucosa
                        glucosa['value'])
    y_fotos_wpp = np.interp(pd.to_numeric(patient_foods_Wapp['fecha_hora']), pd.to_numeric(glucosa['date_time']),  # interpolar fecha de la foto con glucosa
                        glucosa['value'])
    y_fotos_recalls = np.interp(pd.to_numeric(patient_foods_recalls['fecha_hora']), pd.to_numeric(glucosa['date_time']),  # interpolar fecha de la foto con glucosa
                                glucosa['value'])
    
    # valores del dropdown y separarlos para convertirlos en una lista
    if rango_str == '0':
        fig.add_trace(go.Scatter(x=patient_foods_app['FH_reportada'], y=y_foods, mode='markers', name = 'Comida completa', marker=dict(color='red', size=10)))
        fig.add_trace(go.Scatter(x=patient_foods_app['FH_foto'], y=y_foto, mode='markers', name = 'Foto app', marker=dict(color='yellow', size=8)))
        fig.add_trace(go.Scatter(x=patient_foods_Wapp['fecha_hora'], y=y_fotos_wpp, mode='markers', name = 'Whatsapp', marker=dict(color='blue', size=8)))
        fig.add_trace(go.Scatter(x=patient_foods_recalls['fecha_hora'], y=y_fotos_recalls, mode='markers', name = 'Recordatorios', marker=dict(color='purple', size=8)))           
        return fig
    
    elif rango_str == '[>120]':
         rango = [120, float('inf')]
    else:
        rango_split = rango_str.strip('[]').split('-')
        rango = [int(rango_split[0]), int(rango_split[1])]            # rango convertido a lista
    distancias_rango = distancias_df[(distancias_df['Distancia'] >= min(rango)) & (distancias_df['Distancia'] <= max(rango))]    #elegir las comidas que se encuentran en el rango seleccionado
    fotos_wfoods = distancias_rango.set_index('FH_reportada')['FH_foto']
    if distancias_rango.empty:
        return fig 
    y_distancias = np.interp(pd.to_numeric(distancias_rango['FH_reportada']), pd.to_numeric(glucosa['date_time']), glucosa['value']) #interpolar la fecha de las distancias con glucosa
    y_foto_wfood = np.interp(pd.to_numeric(fotos_wfoods.values), pd.to_numeric(glucosa['date_time']), glucosa['value'])
    fig.add_trace(go.Scatter(x=distancias_rango['FH_reportada'], 
                             y=y_distancias, 
                             mode='markers', 
                             name='Comida completa', 
                             marker=dict(color='red', size= 10)))
    fig.add_trace(go.Scatter(x=fotos_wfoods.values, y=y_foto_wfood, mode='markers', name = 'Foto app', marker=dict(color='yellow', size=8)))
    fig.add_trace(go.Scatter(x=patient_foods_Wapp['fecha_hora'], y=y_fotos_wpp, mode='markers', name = 'Foto whatsapp', marker=dict(color='purple', size=8)))    
    fig.update_layout(transition_duration=500)

    return fig

if __name__ == "__main__":
    app.run_server(debug=True, port=8051, host='0.0.0.0')
