import pandas as pd
from dash import Dash, Input, Output, dcc, html
import plotly.graph_objects as go
import plotly.express as px
from plotly.subplots import make_subplots
import requests
import numpy as np
import flask
import test_ch_2_b



def get_visit_data(visitnumber):
    url_visitas = url + "/visits/" + str(visitnumber) + "/patients-data"
    yaisons = requests.get(url_visitas).json()["patient_data"]
    df = pd.json_normalize(yaisons)
    df.drop(["tolerance_curve_measure","name","patient_visit_id", "visit_date"], inplace=True, axis=1)
    df.drop(["sex", "age", "height",], inplace=True, axis=1) # estas ya vienen en 'pacientes'.
    lista = [ { cf["controlled_food"]: cf["increment_auc"] for cf in yaison["tolerance_curve_measure"] } for yaison in yaisons ] 
    df.rename(columns= lambda x: x.split(".")[1] if "." in x else x, inplace=True)
    df["muestra"] = df["patient_visit_id"] 
    df = pd.concat([df, pd.DataFrame(lista)], axis=1)
    return df.loc[df["include_in_analysis"]]


def get_secuenciaciones_data(visitnumber=1):
    secuenciaciones = pd.read_json(f'{url}/microbiota/7/organisms-quantification/{visitnumber}')
    numero_de = secuenciaciones["patient_id"].unique().size
    return numero_de


server = flask.Flask(__name__)

# algunos parámetros globales

url = 'https://nutricion.c3.unam.mx/nd/'
Visitas_e = 4
Comidas_totales_e = 42
Glucometro_e = 14

# fuentes de datos
# consulta online de la población y selección de los que si participaron.
reclutados = pd.read_json(url+"patients/")
reclutados.loc[~reclutados["include_in_analysis"], "finished_periods"] = 0
participantes = reclutados.loc[reclutados["include_in_analysis"]]
excluidos = reclutados.loc[~reclutados["include_in_analysis"],"id"]


# la info de glucómetro y comidas viene de clickhouse_connect.
comidas_df = test_ch_2_b.foods()
excluye = comidas_df.loc[comidas_df["patient"].isin(excluidos)].index
comidas_df.drop(labels=excluye, axis=0, inplace=True)
# creamos un dataframe que tiene como indice (patient, etapa) con el número respectivo de comidas.
resumen_comidas = comidas_df.groupby(["patient","etapas"]).count()["Source"]

glucometro_df = test_ch_2_b.glucometros()
excluye = glucometro_df.loc[glucometro_df["patient_id"].isin(excluidos)].index
glucometro_df.drop(labels=excluye, axis=0, inplace=True)


# consulta online a la API para las visitas, que proporcionan la info de
# visitas y microbiota. la función regresa info solo de participantes incluidos
visitas_datos = [ get_visit_data(visita) for visita in range(1, Visitas_e + 1)]
num_secuencias = [get_secuenciaciones_data(visita) for visita in [1,2,3,4]]


# grafica de tres pies

tres_pies = make_subplots(rows=1, cols=3, specs=[[{"type":"domain"},{"type": "domain"}, {"type": "domain"}]], subplot_titles=["Pacientes por etapas terminadas", "Comidas por etapa", "Comidas por modo de recolección"])
tres_pies.add_trace(
    go.Pie(values=reclutados["finished_periods"].value_counts(), labels=reclutados["finished_periods"].value_counts().index),
    col=1, row=1
)
tres_pies.add_trace(
    go.Pie(values=comidas_df["etapas"].value_counts(), labels=comidas_df["etapas"].value_counts().index),
    col=2, row=1
)
tres_pies.add_trace(
    go.Pie(values=comidas_df["Source"].value_counts(), labels=comidas_df["Source"].value_counts().index),
    col=3, row=1
)
tres_pies.update_traces(textinfo='value+label+percent')
tres_pies.update_layout(showlegend=False)

external_stylesheets = [
    {
        "href": (
            "https://fonts.googleapis.com/css2?"
            "family=Lato:wght@400;700&display=swap"
        ),
        "rel": "stylesheet",
    },
]

app = Dash(__name__, external_stylesheets=external_stylesheets, server=server, url_base_pathname='/nutri/inventario/')
app.title = "NutrINDmex. Inventario de datos"

app.layout = html.Div(children=[
    #encabezado
    html.Div(children=[
        html.P(children="🥑", className="header-emoji"),
        html.H1(children="Inventario de datos", className="header-title"),
        html.P(children=(str(len(reclutados.index)) + " participantes fueron reclutados."),className="header-description"),
        html.P(children=(str(len(participantes.index)) + " están incluidos en el estudio."),className="header-description"),
    ],
    className="header",),
    html.Div(children=[
        html.Div(children=[
            html.H1("Datos Generales", className="header-title2"),
            dcc.Graph(figure=tres_pies)                
        ], className="wrapper"),
    ], className="wrapper"),
    # cobertura por fuente de datos, working
    # es una sucesión de divs.
    # el titulo.
    html.Div(children=[ html.H1("Cobertura de pacientes por fuente de datos", className="header-title2") ]),
    # dos columnas: un dropdown en cada una, con su respectivo título.
    html.Div(children=[
        html.Div(children=[
            html.P("Etapa", className="menu-title"),
            dcc.Dropdown(
                id="etapas_serie",
                options=[
                    {"label":1, "value":1},
                    {"label":2, "value":2},
                    {"label":3, "value":3},
                ],
                value=1,
                clearable=False,
                searchable=False,
                className="dropdown",
            ),      
        ],className="wrapper"),
        html.Div(children=[
            html.P(children="Distribución", className="menu-title"),
            dcc.Dropdown(
                id="distribucion_serie",
                options=[
                    {"label":"Comidas por paciente", "value":1},
                    {"label":"Días de glucómetro por paciente", "value":2},
                ],
                value=1,
                clearable=False,
                searchable=False,
                className="dropdown",
            ),            
        ],className="wrapper"),
    ],style={'display':'flex', 'flexDirection':'row','justify-content': 'center',
        'align-items': 'center'},
    className="wrapper"),
    # la gráfica
    html.Div(children=[dcc.Graph(id="cover_series")],className="wrapper"),
    # el dropdwon de comidas
    html.Div(children=[
        html.P(children="Número mínimo de comidas", className="menu-title"),
        dcc.Slider(
            id="min_comidas_serie",
            min=1,
            max=Comidas_totales_e,
            value=15,
            marks={1:"1",5:"5",10:"10",15:"15", 21:"21",30:"30", Comidas_totales_e:str(Comidas_totales_e)}
        ),            
    ],className="wrapper"),
    # el dropdown de glucómetro
    html.Div(children=[
        html.P(children="Númer mínimo de días con glucómetro", className="menu-title"),                        
        dcc.Slider(
            id="min_glucosa_serie",
            min=1,
            max=Glucometro_e,
            value=7,
            marks={1:"1", 5:"5", 7:"7", 10:"10",14:"14"}, className="slider",
        ),
    ],className="wrapper"),
    # cobertura de pacientes por prueba. working.
    html.Div(children=[
        html.H1("Cobertura de pacientes por protocolo.", className="header-title2"),
        # selector de etapa
        html.Div(children=[
            html.Div(children="Etapa", className="menu-title"),
            dcc.Dropdown(
                id="select_vis_protocolo",
                options=[
                    {"label":1, "value":1},
                    {"label":2, "value":2},
                    {"label":3, "value":3},
                    {"label":4, "value":4}
                ],
                value=1,
                clearable=False,
                searchable=False,
                className="dropdown",
            ),
        ], className="wrapper"),
        #y gráfica
        dcc.Graph(id="cobertura_protocolo"),
    ], className="wrapper"),
    #heatmap
    html.Div(children=[
        html.H1("Pruebas por paciente", className="header-title2"),
        # selector de etapa
        html.Div(children=[
            html.Div(children="Etapa", className="menu-title"),
            dcc.Dropdown(
                id="select_vis_pppaciente",
                options=[
                    {"label":1, "value":1},
                    {"label":2, "value":2},
                    {"label":3, "value":3},
                    {"label":4, "value":4}
                ],
                value=1,
                clearable=False,
                searchable=False,
                className="dropdown",
            ),
        ], className="wrapper"),
        #y gráfica
        dcc.Graph(id="pppaciente"),
    ], className="wrapper"),
]) # cierre del layout

@app.callback(
     Output("cover_series", "figure"),
     Input("etapas_serie","value"),
     Input("min_comidas_serie", "value"),
     Input("min_glucosa_serie", "value"),
     Input("distribucion_serie", "value")
)
def update_cobertura_series(etapaSel=1, min_comidas=10, min_glucosa=7, distribucion=1):
    gauges = {
            'shape': "bullet",
            'axis': {'range': [None, 100]},
            'steps': [
                {'range': [0, 25], 'color': "lightcyan"},
                {'range': [25, 50], 'color': "lightblue"},                
                {'range': [50, 75], 'color': "skyblue"},
                {'range': [75, 100], 'color': "steelblue"}],            
            'bar': {'color': "blue"},
        }
    numbers = { 'suffix':'%', 'valueformat':'.2f'}
    modes = "number+gauge"

    comidas_data = resumen_comidas.loc[(slice(None),f'e{etapaSel}')] > min_comidas
    glucosa_data = glucometro_df.loc[glucometro_df["visit_id"] == etapaSel, "amount_of_days"] > min_glucosa
    microbiota_data = num_secuencias[etapaSel-1]
    if distribucion == 1:
        xs = resumen_comidas.loc[(slice(None),f'e{etapaSel}')].sort_values()
        title = f"Comidas por paciente en la etapa {etapaSel}"
        medCadena = 'comidas'
        minimo = min_comidas
    if distribucion == 2:
        xs = glucometro_df.loc[glucometro_df["visit_id"] == etapaSel]["amount_of_days"].sort_values()
        title = f'dias de glucómetro por paciente en la etapa {etapaSel}'
        medCadena = 'días de glucómetro'
        minimo=min_glucosa
    xs.index = xs.index.astype(str)
    
# La N de la etapa debe ser el número de participantes que cumplieron la etapa.
# esto es diferente al número de participantes que asistieron a la visita N o N+1.
    N_etapa = len(participantes.loc[participantes["finished_periods"].isin(range(etapaSel, 4))].index)
    multi = make_subplots(rows=2, cols=1, specs=[[{"type":"indicator"}],[{}]])
    multi.add_trace(go.Indicator(
       value = 100 * comidas_data.sum() / N_etapa,
        domain = {'x': [0.25, 1], 'y': [0.70, 0.8]},
        title = {'text': "Comidas"},
        mode = modes,
        number = numbers,
        gauge = gauges,
    ))
    multi.add_trace(go.Indicator(
        value = 100 * glucosa_data.sum() / N_etapa,
        domain = {'x': [0.25, 1], 'y': [0.9, 1.0]},
        title = {'text': "Glucómetro"},
        mode = modes,
        number = numbers,
        gauge = gauges
    ))
    multi.add_trace(go.Indicator(
        value = 100 * microbiota_data / N_etapa,
        domain = {'x': [0.25, 1], 'y': [0.5, 0.6]},
        title = {'text': "Microbiota"},
        mode = modes,
        number = numbers,
        gauge = gauges
    ))
    multi.add_trace(
        go.Scatter(x=xs.index, y=xs)
    )
    multi.update_xaxes(title_text=f'{N_etapa} pacientes', row=2, col=1)
    multi.update_yaxes(title_text=medCadena, row=2, col=1)
    multi.add_shape(go.layout.Shape(type='line', yref='y1', xref='x1', x0=0, x1=1000, xsizemode='pixel', y0=minimo, y1=minimo,
                           line=dict(color='red', width=1, dash='dot')))
    
    return multi




@app.callback(
    Output("cobertura_protocolo","figure"),
    Input("select_vis_protocolo", "value")
)
def update_por_protocolo(visita):
    lcol = 1
    lrow = 1
    df1 = visitas_datos[visita-1]
    protocolos = {
        "pyhsiological" : ["pas", "pad", "ipaq", "stress", "hadsa", "hadsd"],
        "tolerance_curves" : ["Pan", "Gelatina", "Glucosa"],
        "corporal_composition" : ["mmkg", "mmekg", "mgkg", "mlgkg", "percent_mg", "gv"],
        "biochemical" :["hba1c", "glucose", "ct", "tag", "ldl", "hdl", "pcr","alt", "ast","homa", "insuline" ],
        "antropometric" : ["imc", "weight", "waist_circumference"]
    }
    figura = make_subplots(rows=2, cols=3, vertical_spacing=0.25, subplot_titles=list(protocolos.keys()))
    for protocolo in protocolos.keys():
        columnas = list(set(df1.columns).intersection(set(protocolos[protocolo])))
        cuentas = 100 * df1[columnas].replace("",np.nan).count() / len(df1)
        figura.add_trace(
            go.Bar(x=cuentas.index, y=cuentas, textposition='auto', text=cuentas),
            row = lrow,
            col = lcol,
        )
        lcol = lcol + 1
        if lcol > 3:
            lcol = 1
            lrow = lrow + 1
    figura.update_layout(showlegend=False)
    figura.update_traces(texttemplate='%{text:.2s}', textposition='auto')
    return figura

@app.callback(
    Output("pppaciente","figure"),
    Input("select_vis_pppaciente", "value")
)
def update_por_prueba(visita):
    lcol = 1
    lrow = 1
    df1 = visitas_datos[visita-1]
    protocolos = {
        "pyhsiological" : ["pas", "pad", "ipaq", "stress", "hadsa", "hadsd"],
        "tolerance_curves" : ["Pan", "Gelatina", "Glucosa"],
        "corporal_composition" : ["mmkg", "mmekg", "mgkg", "mlgkg", "percent_mg", "gv"],
        "biochemical" :["hba1c", "glucose", "ct", "tag", "ldl", "hdl", "pcr","alt", "ast","homa", "insuline" ],
        "antropometric" : ["imc", "weight", "waist_circumference"]
    }
    heatmap = df1.set_index("id").notna().astype(int)
    colorscale = [[0, 'gold'], [1,'mediumturquoise']]
    columnas = [ test  for protocolo in protocolos.keys() for test in protocolos[protocolo]]
    lista = list(set(df1.columns).intersection(set(columnas)))
    figura = px.imshow(heatmap[lista], color_continuous_scale=colorscale)
    figura.update_layout(dict(coloraxis_showscale=False))
    return figura


if __name__ == "__main__":
    app.run_server(debug=True, port=8090, host='0.0.0.0')
